<?php
/*--------------------------------------------------------------------
 OnGetSellingUnitWeightEventListener.php 2020-2-24
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2020 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Shop\Properties\SellingUnitWeight\Database\Listener;

use Gambio\Shop\Properties\SellingUnitImages\Database\Exceptions\ModifierCollectionDoesNotContainAnyPropertiesException;
use Gambio\Shop\Properties\SellingUnitImages\Database\Repository\Helpers\CombisIdIdentifierInterface;
use Gambio\Shop\Properties\SellingUnitWeight\Database\Repository\DTO\CombisIdDto;
use Gambio\Shop\Properties\SellingUnitWeight\Database\Service\ReadServiceInterface;
use Gambio\Shop\Properties\SellingUnitWeight\Exceptions\CombisIdDoesNotExistsException;
use Gambio\Shop\SellingUnit\Database\Unit\Events\Interfaces\OnGetSellingUnitWeightEventInterface;
use Gambio\Shop\SellingUnit\Unit\ValueObjects\Weight;

/**
 * Class OnGetSellingUnitWeightEventListener
 * @package Gambio\Shop\Properties\SellingUnitWeight\Database\Listener
 */
class OnGetSellingUnitWeightEventListener
{
    /**
     * @var ReadServiceInterface
     */
    protected $service;
    
    /**
     * @var CombisIdIdentifierInterface
     */
    protected $combisIdIdentifier;
    
    
    /**
     * OnGetSellingUnitWeightEventListener constructor.
     *
     * @param ReadServiceInterface        $service
     * @param CombisIdIdentifierInterface $combisIdIdentifier
     */
    public function __construct(ReadServiceInterface $service, CombisIdIdentifierInterface $combisIdIdentifier)
    {
        $this->service            = $service;
        $this->combisIdIdentifier = $combisIdIdentifier;
    }
    
    
    /**
     * @param OnGetSellingUnitWeightEventInterface $event
     */
    public function __invoke(OnGetSellingUnitWeightEventInterface $event)
    {
        try {
            $combisId    = $this->combisIdIdentifier->getCombisId($event->productId(), $event->modifiers())->combisId();
            $combisIdDto = new CombisIdDto($combisId);
            
            $combisWeight = $this->service->combinationWeight($combisIdDto)->weight();
        } catch (CombisIdDoesNotExistsException | ModifierCollectionDoesNotContainAnyPropertiesException $exception) {
            
            unset($exception);
            
            return;
        }
        
        $productWeight     = $event->weight()->value();
        $useCombisWeight   = $event->product()->usePropertiesCombisWeight();
        $sellingUnitWeight = $useCombisWeight ? $combisWeight : $productWeight + $combisWeight;
        
        $event->setWeight(new Weight($sellingUnitWeight, $event->product()->showWeight()));
    }
}